/*
 *  SPDX-FileCopyrightText: 2004-2006 Cyrille Berger <cberger@cberger.net>
 *
 *  SPDX-License-Identifier: GPL-2.0-or-later
 */
#ifndef COLORSPACE_GRAYSCALE_F32_H_
#define COLORSPACE_GRAYSCALE_F32_H_

#include "LcmsColorSpace.h"
#include <KLocalizedString>
#include <KoColorModelStandardIds.h>

#define TYPE_GRAYA_FLT (FLOAT_SH(1) | COLORSPACE_SH(PT_GRAY) | EXTRA_SH(1) | CHANNELS_SH(1) | BYTES_SH(4))

struct KoGrayF32Traits;

class GrayF32ColorSpace : public LcmsColorSpace<KoGrayF32Traits>
{
public:
    GrayF32ColorSpace(const QString &name, KoColorProfile *p);

    bool willDegrade(ColorSpaceIndependence) const override
    {
        return false;
    }

    KoID colorModelId() const override
    {
        return GrayAColorModelID;
    }

    KoID colorDepthId() const override
    {
        return Float32BitsColorDepthID;
    }

    virtual KoColorSpace *clone() const;

    void colorToXML(const quint8 *pixel, QDomDocument &doc, QDomElement &colorElt) const override;

    void colorFromXML(quint8 *pixel, const QDomElement &elt) const override;

    void toHSY(const QVector<double> &channelValues, qreal *hue, qreal *sat, qreal *luma) const override;
    QVector<double> fromHSY(qreal *hue, qreal *sat, qreal *luma) const override;
    void toYUV(const QVector<double> &channelValues, qreal *y, qreal *u, qreal *v) const override;
    QVector<double> fromYUV(qreal *y, qreal *u, qreal *v) const override;

    static QString colorSpaceId()
    {
        return "GRAYAF32";
    }

    bool hasHighDynamicRange() const override
    {
        return true;
    }
};

class GrayF32ColorSpaceFactory : public LcmsColorSpaceFactory
{
public:
    GrayF32ColorSpaceFactory()
        : LcmsColorSpaceFactory(TYPE_GRAYA_FLT, cmsSigGrayData)
    {
    }

    QString id() const override
    {
        return GrayF32ColorSpace::colorSpaceId();
    }

    QString name() const override
    {
        return i18n("Grayscale/Alpha (32-bit float/channel)");
    }

    KoID colorModelId() const override
    {
        return GrayAColorModelID;
    }

    KoID colorDepthId() const override
    {
        return Float32BitsColorDepthID;
    }

    int referenceDepth() const override
    {
        return 32;
    }

    bool userVisible() const override
    {
        return true;
    }

    KoColorSpace *createColorSpace(const KoColorProfile *p) const override
    {
        return new GrayF32ColorSpace(name(), p->clone());
    }

    QString defaultProfile() const override
    {
        return "gray built-in";
    }

    bool isHdr() const override
    {
        return true;
    }
};

#endif // KIS_STRATEGY_COLORSPACE_GRAYSCALE_H_
